<?php
header('Content-Type: application/json');
require_once 'config.php';

try {
    $conn = new PDO("mysql:host=$host;dbname=$dbname", $username, $password);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $action = $_GET['action'] ?? 'list';

    // Ações que não exigem autenticação
    if ($action === 'get_item_by_barcode') {
        $barcode = $_GET['barcode'] ?? '';
        if (empty($barcode)) {
            http_response_code(400);
            echo json_encode(['error' => 'Código de barras não fornecido']);
            exit;
        }
        $stmt = $conn->prepare("
            SELECT si.id, si.name, sib.barcode, si.quantity, si.min_quantity 
            FROM stock_items si
            JOIN stock_item_barcodes sib ON si.id = sib.item_id
            WHERE sib.barcode = ?
        ");
        $stmt->execute([$barcode]);
        $item = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$item) {
            http_response_code(404);
            echo json_encode(['error' => 'Equipamento não encontrado para o código de barras']);
            exit;
        }
        echo json_encode($item);
        exit;
    } elseif ($action === 'get_client_by_cpf') {
        $cpf = $_GET['cpf'] ?? '';
        if (empty($cpf)) {
            http_response_code(400);
            echo json_encode(['error' => 'CPF não fornecido']);
            exit;
        }
        // Normalizar CPF (remover pontos e traços)
        $cpf = preg_replace('/[\.\-]/', '', $cpf);
        $stmt = $conn->prepare("SELECT cpf, name FROM clients WHERE cpf = ?");
        $stmt->execute([$cpf]);
        $client = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$client) {
            http_response_code(404);
            echo json_encode(['error' => 'Cliente não encontrado para o CPF']);
            exit;
        }
        echo json_encode($client);
        exit;
    } elseif ($action === 'add_movement' && $_SERVER['REQUEST_METHOD'] === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        if (!isset($input['item_id']) || !isset($input['installer']) || !isset($input['type']) || !isset($input['quantity']) || !isset($input['client_id'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Parâmetros item_id, installer, type, quantity e client_id são obrigatórios']);
            exit;
        }
        // Normalizar CPF
        $client_id = preg_replace('/[\.\-]/', '', $input['client_id']);
        $stmt = $conn->prepare("SELECT cpf FROM clients WHERE cpf = ?");
        $stmt->execute([$client_id]);
        if (!$stmt->fetch(PDO::FETCH_ASSOC)) {
            http_response_code(400);
            echo json_encode(['error' => 'Cliente não encontrado para o CPF informado']);
            exit;
        }
        $conn->beginTransaction();
        if ($input['type'] === 'entry') {
            $stmt = $conn->prepare("UPDATE stock_items SET quantity = quantity + ? WHERE id = ?");
            $stmt->execute([$input['quantity'], $input['item_id']]);
        } elseif ($input['type'] === 'exit') {
            $stmt = $conn->prepare("SELECT quantity FROM stock_items WHERE id = ?");
            $stmt->execute([$input['item_id']]);
            $currentQuantity = $stmt->fetch(PDO::FETCH_ASSOC)['quantity'];
            if ($currentQuantity < $input['quantity']) {
                $conn->rollBack();
                http_response_code(400);
                echo json_encode(['error' => 'Quantidade insuficiente no estoque']);
                exit;
            }
            $stmt = $conn->prepare("UPDATE stock_items SET quantity = quantity - ? WHERE id = ?");
            $stmt->execute([$input['quantity'], $input['item_id']]);
        } else {
            $conn->rollBack();
            http_response_code(400);
            echo json_encode(['error' => 'Tipo de movimentação inválido']);
            exit;
        }
        $stmt = $conn->prepare("INSERT INTO stock_movements (item_id, installer, client_id, type, quantity) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$input['item_id'], $input['installer'], $client_id, $input['type'], $input['quantity']]);
        $conn->commit();
        echo json_encode(['success' => true]);
        exit;
    }

    // Ações que exigem autenticação
    session_start();
    if (!isset($_SESSION['user_id'])) {
        http_response_code(401);
        echo json_encode(['error' => 'Não autenticado']);
        error_log('Erro: Acesso não autenticado a stock.php');
        exit;
    }

    if ($action === 'check_auth') {
        echo json_encode(['success' => true, 'user_id' => $_SESSION['user_id']]);
        exit;
    } elseif ($action === 'list_items') {
        $stmt = $conn->query("
            SELECT si.id, si.name, GROUP_CONCAT(sib.barcode) as barcodes, si.quantity, si.min_quantity 
            FROM stock_items si
            LEFT JOIN stock_item_barcodes sib ON si.id = sib.item_id
            GROUP BY si.id
            ORDER BY si.name
        ");
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($items);
        exit;
    } elseif ($action === 'get_client_by_serial') {
        $serial = $_GET['serial'] ?? '';
        if (empty($serial)) {
            http_response_code(400);
            echo json_encode(['error' => 'Serial não fornecido']);
            exit;
        }
        $stmt = $conn->prepare("SELECT cpf, name, serial FROM clients WHERE serial = ?");
        $stmt->execute([$serial]);
        $client = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$client) {
            http_response_code(404);
            echo json_encode(['error' => 'Cliente não encontrado para o serial']);
            exit;
        }
        echo json_encode($client);
        exit;
    } elseif ($action === 'add_item' && $_SERVER['REQUEST_METHOD'] === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        if (!isset($input['name']) || !isset($input['quantity']) || !isset($input['min_quantity']) || !isset($input['barcodes']) || !is_array($input['barcodes'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Parâmetros name, quantity, min_quantity e barcodes (array) são obrigatórios']);
            exit;
        }
        $conn->beginTransaction();
        $stmt = $conn->prepare("INSERT INTO stock_items (name, quantity, min_quantity) VALUES (?, ?, ?)");
        $stmt->execute([$input['name'], $input['quantity'], $input['min_quantity']]);
        $itemId = $conn->lastInsertId();
        $stmt = $conn->prepare("INSERT INTO stock_item_barcodes (item_id, barcode) VALUES (?, ?)");
        foreach ($input['barcodes'] as $barcode) {
            $barcode = trim($barcode);
            if (!empty($barcode)) {
                $stmt->execute([$itemId, $barcode]);
            }
        }
        $stmt = $conn->prepare("INSERT INTO stock_movements (item_id, type, quantity) VALUES (?, 'entry', ?)");
        $stmt->execute([$itemId, $input['quantity']]);
        $conn->commit();
        echo json_encode(['success' => true, 'item_id' => $itemId]);
        exit;
    } elseif ($action === 'get_movements_by_installer') {
        $installer = $_GET['installer'] ?? '';
        $stmt = $conn->prepare("
            SELECT m.id, m.item_id, i.name as item_name, m.installer, m.client_id, c.name as client_name, m.type, m.quantity, m.movement_date
            FROM stock_movements m
            JOIN stock_items i ON m.item_id = i.id
            LEFT JOIN clients c ON m.client_id = c.cpf
            WHERE m.installer = ? OR ? = ''
            ORDER BY m.movement_date DESC
            LIMIT 100
        ");
        $stmt->execute([$installer, $installer]);
        $movements = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($movements);
        exit;
    } elseif ($action === 'list') {
        $stmt = $conn->query("
            SELECT si.id, si.name, GROUP_CONCAT(sib.barcode) as barcodes, si.quantity, si.min_quantity 
            FROM stock_items si
            LEFT JOIN stock_item_barcodes sib ON si.id = sib.item_id
            GROUP BY si.id
            ORDER BY si.name
        ");
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $stmt = $conn->query("
            SELECT m.id, m.item_id, i.name as item_name, m.installer, m.client_id, c.name as client_name, m.type, m.quantity, m.movement_date
            FROM stock_movements m
            JOIN stock_items i ON m.item_id = i.id
            LEFT JOIN clients c ON m.client_id = c.cpf
            ORDER BY m.movement_date DESC
            LIMIT 100
        ");
        $movements = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $stmt = $conn->query("
            SELECT installer, COUNT(*) as count
            FROM stock_movements
            GROUP BY installer
            ORDER BY count DESC
            LIMIT 5
        ");
        $movementsByInstaller = $stmt->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode([
            'items' => $items,
            'movements' => $movements,
            'movementsByInstaller' => $movementsByInstaller
        ]);
        exit;
    }

    http_response_code(400);
    echo json_encode(['error' => 'Ação inválida']);
} catch (PDOException $e) {
    http_response_code(500);
    $errorMessage = 'Erro no banco de dados: ' . $e->getMessage();
    error_log($errorMessage);
    echo json_encode(['error' => $errorMessage]);
} catch (Exception $e) {
    http_response_code(500);
    $errorMessage = 'Erro inesperado: ' . $e->getMessage();
    error_log($errorMessage);
    echo json_encode(['error' => $errorMessage]);
}
?>