<?php
header('Content-Type: application/json');
require_once 'config.php';

session_start();

try {
    // Verificar autenticação
    if (!isset($_SESSION['user_id'])) {
        http_response_code(401);
        echo json_encode(['error' => 'Não autenticado']);
        exit;
    }

    // Conectar ao banco
    $conn = new PDO("mysql:host=$host;dbname=$dbname", $username, $password);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Buscar clientes
    $stmt = $conn->prepare("
        SELECT c.*, p.name AS plan_name
        FROM clients c
        LEFT JOIN plans p ON c.planId = p.id
    ");
    $stmt->execute();
    $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Funções de formatação
    function formatCPF($cpf) {
        if (!$cpf) return '';
        $cpf = preg_replace('/\D/', '', $cpf);
        return preg_replace('/(\d{3})(\d{3})(\d{3})(\d{2})/', '$1.$2.$3-$4', $cpf);
    }

    function formatPhone($phone) {
        if (!$phone) return '';
        $phone = preg_replace('/\D/', '', $phone);
        if (strlen($phone) === 11) {
            return preg_replace('/(\d{2})(\d{5})(\d{4})/', '($1) $2-$3', $phone);
        } else if (strlen($phone) === 10) {
            return preg_replace('/(\d{2})(\d{4})(\d{4})/', '($1) $2-$3', $phone);
        }
        return $phone;
    }

    function formatDate($date) {
        if (!$date) return '';
        $parts = explode('-', $date);
        return $parts[2] . '/' . $parts[1] . '/' . $parts[0];
    }

    // Gerar conteúdo LaTeX
    $latexContent = "
\\documentclass[a4paper,landscape]{article}
\\usepackage[utf8]{inputenc}
\\usepackage[T1]{fontenc}
\\usepackage{geometry}
\\geometry{left=1cm,right=1cm,top=1cm,bottom=1cm}
\\usepackage{longtable}
\\usepackage{booktabs}
\\usepackage{pdflscape}
\\usepackage{xcolor}
\\usepackage{colortbl}
\\definecolor{headerblue}{RGB}{30,64,175}
\\definecolor{rowgray}{RGB}{245,245,245}
\\begin{document}
\\begin{center}
  \\LARGE\\textbf{Relatório de Clientes - Sistema ERP}\\\\
  \\small Gerado em: \\today
\\end{center}
\\vspace{0.5cm}
\\begin{longtable}{|p{2.5cm}|p{2.5cm}|p{4cm}|p{5cm}|p{2cm}|p{2cm}|p{2cm}|p{1.5cm}|p{2cm}|p{2cm}|p{3cm}|p{2cm}|p{2cm}|p{2cm}|p{2cm}|}
  \\hline
  \\rowcolor{headerblue}
  \\color{white}\\textbf{CPF} &
  \\color{white}\\textbf{Serial} &
  \\color{white}\\textbf{Nome} &
  \\color{white}\\textbf{Endereço} &
  \\color{white}\\textbf{Número} &
  \\color{white}\\textbf{Compl.} &
  \\color{white}\\textbf{Cidade} &
  \\color{white}\\textbf{Venc.} &
  \\color{white}\\textbf{Telefone} &
  \\color{white}\\textbf{Nasc.} &
  \\color{white}\\textbf{Observação} &
  \\color{white}\\textbf{Plano} &
  \\color{white}\\textbf{Instalador} &
  \\color{white}\\textbf{PPPoE} &
  \\color{white}\\textbf{Senha} \\\\
  \\hline
  \\endhead
";

    foreach ($clients as $index => $client) {
        $rowColor = $index % 2 === 0 ? '\\rowcolor{rowgray}' : '';
        $latexContent .= "
  $rowColor
  " . formatCPF($client['cpf']) . " &
  " . ($client['serial'] ? str_replace('&', '\\&', $client['serial']) : '') . " &
  " . ($client['name'] ? str_replace('&', '\\&', $client['name']) : '') . " &
  " . ($client['address'] ? str_replace('&', '\\&', $client['address']) : '') . " &
  " . ($client['number'] ?? '') . " &
  " . ($client['complement'] ?? '') . " &
  " . ($client['city'] ?? '') . " &
  " . ($client['dueDay'] ?? '') . " &
  " . formatPhone($client['phone'] ?? '') . " &
  " . formatDate($client['birthDate'] ?? '') . " &
  " . ($client['observation'] ? str_replace(['&', '\n'], ['\\&', ' '], $client['observation']) : '') . " &
  " . ($client['plan_name'] ? str_replace('&', '\\&', $client['plan_name']) : '') . " &
  " . ($client['installer'] ? str_replace('&', '\\&', $client['installer']) : '') . " &
  " . ($client['pppoe'] ? str_replace('&', '\\&', $client['pppoe']) : '') . " &
  " . ($client['password'] ? str_replace('&', '\\&', $client['password']) : '') . " \\\\
  \\hline
";
    }

    $latexContent .= "
  \\end{longtable}
\\end{document}
";

    // Salvar o arquivo LaTeX temporário
    $tempDir = sys_get_temp_dir();
    $latexFile = $tempDir . '/report.tex';
    $pdfFile = $tempDir . '/report.pdf';
    file_put_contents($latexFile, $latexContent);

    // Compilar o LaTeX em PDF usando latexmk
    $command = "latexmk -pdf -outdir=$tempDir $latexFile 2>&1";
    exec($command, $output, $returnVar);

    if ($returnVar !== 0) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao gerar PDF: ' . implode("\n", $output)]);
        exit;
    }

    // Enviar o PDF como download
    if (file_exists($pdfFile)) {
        header('Content-Type: application/pdf');
        header('Content-Disposition: attachment; filename="relatorio_clientes.pdf"');
        header('Content-Length: ' . filesize($pdfFile));
        readfile($pdfFile);

        // Limpar arquivos temporários
        unlink($latexFile);
        unlink($pdfFile);
        array_map('unlink', glob("$tempDir/report.*"));
        exit;
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'PDF não foi gerado']);
        exit;
    }

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Erro no banco de dados: ' . $e->getMessage()]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
?>